// $Id: CTextStream.cpp,v 1.6 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CTextStream.hpp"
#include <iomanip>

using Exponent::IO::CTextStream;
using std::ios;
using std::endl;
using std::setprecision;
using std::setiosflags;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CTextStream, CCountedObject);

//	===========================================================================
CTextStream::CTextStream(const CSystemString &path, const EStreamMode mode)
		   : m_streamIsOpen(false)
		   , m_mode(e_input)
{
	EXPONENT_CLASS_CONSTRUCTION(CTextStream);
	strcpy(m_buffer, "Empty string");
	this->openStream(path, mode);
}

//	===========================================================================
CTextStream::CTextStream()
		   : m_streamIsOpen(false)
		   , m_mode(e_input)
{
	EXPONENT_CLASS_CONSTRUCTION(CTextStream);
	strcpy(m_buffer, "Empty string");
}

//	===========================================================================
CTextStream::~CTextStream()
{
	EXPONENT_CLASS_DESTRUCTION(CTextStream);
	this->closeStream();
}

//	===========================================================================
bool CTextStream::openStream(const CSystemString &path, const EStreamMode mode)
{
	// Close the stream
	this->closeStream();

	// Store the mode
	m_mode = mode;

	// Store the filepath
	m_filePath = path;

	// Now open
	if (mode == e_input)
	{
		// Open the file
		m_inputFile.open(path.getString(), ios::in);

		// Are we open?
		m_streamIsOpen = m_inputFile.is_open();
	}
	else if (mode == e_output)
	{
		// Open the file
		m_outFile.open(path.getString(), ios::trunc);

		// Are we open?
		m_streamIsOpen = m_outFile.is_open();
	}

	return m_streamIsOpen;
}

//	===========================================================================
void CTextStream::closeStream()
{
	if (m_mode == e_input && m_streamIsOpen)
	{
		// Close the file
		m_inputFile.close();
	}
	else if (m_mode == e_output)
	{
		// Write any pending data
		m_outFile.flush();

		// Close the file
		m_outFile.close();
	}

	// Are we open?
	m_streamIsOpen = false;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const char *string)
{
	if (this->validForOutput() && string)
	{
		m_outFile.clear();
		m_outFile << string;
		m_outFile.flush();
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const CString &string)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		m_outFile << string.getString();
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const CSystemString &string)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		m_outFile << string.getString();
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const CCountedObject &object)
{
	if (this->validForOutput())
	{
		// Get the description
		char string[256];
		memset(string, 0, 256 * sizeof(char));
		object.getObjectDescription(string, 256);

		m_outFile.clear();

		// Stream it out
		m_outFile << string;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const long value)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		m_outFile << value;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const int value)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		m_outFile << value;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const double value)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		m_outFile << setiosflags(ios::fixed | ios::showpoint) << setprecision(20) << value;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const float value)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		m_outFile << setiosflags(ios::fixed | ios::showpoint) << setprecision(20) << value;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator << (const bool value)
{
	if (this->validForOutput())
	{
		m_outFile.clear();
		if (value)
		{
			m_outFile << "true";
		}
		else
		{
			m_outFile << "false";
		}
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (char *string)
{
	if (this->validForInput() && string)
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Store the buffer
		strcpy(string, m_buffer);
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (CString &string)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Store the buffer
		string = m_buffer;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (CSystemString &string)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Store the buffer
		string = m_buffer;
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (long &value)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Conver it to a long
		value = atol(m_buffer);
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (int &value)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Conver it to a long
		value = atoi(m_buffer);
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (double &value)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Conver it to a long
		value = atof(m_buffer);
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (float &value)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer) - 1] == '\r')
		{
			m_buffer[strlen(m_buffer) - 1] = '\0';
		}

		// Conver it to a long
		value = (float)atof(m_buffer);
	}
	return *this;
}

//	===========================================================================
CTextStream &CTextStream::operator >> (bool &value)
{
	if (this->validForInput())
	{
		// Clear the error bits
		m_inputFile.clear();

		// Read the line
		m_inputFile.getline(m_buffer, CTEXTSTREAM_MAX_CHARACTERS, '\n');

		// Strip the M$ \r\n new line
		if (m_buffer[strlen(m_buffer)] == '\r')
		{
			m_buffer[strlen(m_buffer)] = '\0';
		}

		// Store the buffer
		CString theString = m_buffer;

		// Store the value
		value = (theString == "true");
	}
	return *this;
}

//	===========================================================================
bool CTextStream::validForOutput()
{
	return (m_streamIsOpen && m_mode == e_output);
}

//	===========================================================================
bool CTextStream::validForInput()
{
	return (m_streamIsOpen && m_mode == e_input);
}

//	===========================================================================
bool CTextStream::hasReachedEOF()
{
	if (this->validForInput())
	{
		return m_inputFile.eof();
	}
	else if (this->validForOutput())
	{
		// Well you never really reach the eof whilst outputing
		return false;
	}

	// We arent open and therefore are at the end of file
	return true;
}

//	===========================================================================
void CTextStream::flushToDisk()
{
	if (this->validForOutput())
	{
		m_outFile.flush();
	}
}